/**
 * App Chat
 */

'use strict';

document.addEventListener('DOMContentLoaded', function () {
  (function () {
    const chatContactsBody = document.querySelector('.app-chat-contacts .sidebar-body'),
      chatContactListItems = [].slice.call(
        document.querySelectorAll('.chat-contact-list-item:not(.chat-contact-list-item-title)')
      ),
      chatHistoryBody = document.querySelector('.chat-history-body'),
      chatSidebarLeftBody = document.querySelector('.app-chat-sidebar-left .sidebar-body'),
      chatSidebarRightBody = document.querySelector('.app-chat-sidebar-right .sidebar-body'),
      chatUserStatus = [].slice.call(document.querySelectorAll(".form-check-input[name='chat-user-status']")),
      chatSidebarLeftUserAbout = $('.chat-sidebar-left-user-about'),
      formSendMessage = document.querySelector('.form-send-message'),
      messageInput = document.querySelector('.message-input'),
      searchInput = document.querySelector('.chat-search-input'),
      speechToText = $('.speech-to-text'), // ! jQuery dependency for speech to text
      userStatusObj = {
        active: 'avatar-online',
        offline: 'avatar-offline',
        away: 'avatar-away',
        busy: 'avatar-busy'
      };

    // Initialize PerfectScrollbar
    // ------------------------------

    // Chat contacts scrollbar
    if (chatContactsBody) {
      new PerfectScrollbar(chatContactsBody, {
        wheelPropagation: false,
        suppressScrollX: true
      });
    }

    // Chat history scrollbar
    if (chatHistoryBody) {
      new PerfectScrollbar(chatHistoryBody, {
        wheelPropagation: false,
        suppressScrollX: true
      });
    }

    // Sidebar left scrollbar
    if (chatSidebarLeftBody) {
      new PerfectScrollbar(chatSidebarLeftBody, {
        wheelPropagation: false,
        suppressScrollX: true
      });
    }

    // Sidebar right scrollbar
    if (chatSidebarRightBody) {
      new PerfectScrollbar(chatSidebarRightBody, {
        wheelPropagation: false,
        suppressScrollX: true
      });
    }

    // Scroll to bottom function
    function scrollToBottom() {
      chatHistoryBody.scrollTo(0, chatHistoryBody.scrollHeight);
    }
    scrollToBottom();

    // User About Maxlength Init
    if (chatSidebarLeftUserAbout.length) {
      chatSidebarLeftUserAbout.maxlength({
        alwaysShow: true,
        warningClass: 'label label-success bg-success text-white',
        limitReachedClass: 'label label-danger',
        separator: '/',
        validate: true,
        threshold: 120
      });
    }

    // Update user status
    chatUserStatus.forEach(el => {
      el.addEventListener('click', e => {
        let chatLeftSidebarUserAvatar = document.querySelector('.chat-sidebar-left-user .avatar'),
          value = e.currentTarget.value;
        //Update status in left sidebar user avatar
        chatLeftSidebarUserAvatar.removeAttribute('class');
        Helpers._addClass('avatar avatar-xl ' + userStatusObj[value] + '', chatLeftSidebarUserAvatar);
        //Update status in contacts sidebar user avatar
        let chatContactsUserAvatar = document.querySelector('.app-chat-contacts .avatar');
        chatContactsUserAvatar.removeAttribute('class');
        Helpers._addClass('flex-shrink-0 avatar ' + userStatusObj[value] + ' me-3', chatContactsUserAvatar);
      });
    });

    // Select chat or contact
    chatContactListItems.forEach(chatContactListItem => {
      // Bind click event to each chat contact list item
      chatContactListItem.addEventListener('click', e => {
        // Remove active class from chat contact list item
        chatContactListItems.forEach(chatContactListItem => {
          chatContactListItem.classList.remove('active');
        });
        // Add active class to current chat contact list item
        e.currentTarget.classList.add('active');

        // Update sidebar with contact information
        const contactName = e.currentTarget.querySelector('h6').textContent;
        const contactAvatar = e.currentTarget.querySelector('img').src;
        const contactEmail = e.currentTarget.dataset.email;
        const contactPhone = e.currentTarget.dataset.phone;
        const contactHours = e.currentTarget.dataset.hours;
        const contactDescription = e.currentTarget.dataset.description;
        const contactStatus = e.currentTarget.dataset.status;

        document.getElementById('sidebar-contact-name').textContent = contactName;
        document.getElementById('sidebar-contact-avatar').src = contactAvatar;
        document.getElementById('sidebar-contact-email').textContent = contactEmail;
        document.getElementById('sidebar-contact-phone').textContent = contactPhone;
        document.getElementById('sidebar-contact-hours').textContent = contactHours;
        document.getElementById('sidebar-contact-description').textContent = contactDescription;
        document.getElementById('sidebar-contact-status').textContent = contactStatus;

        // Fetch messages for the selected contact
        const clienteId = e.currentTarget.dataset.clienteId;
        fetchMessages(clienteId);
      });
    });

    // Filter Chats
    if (searchInput) {
      searchInput.addEventListener('keyup', e => {
        let searchValue = e.currentTarget.value.toLowerCase(),
          searchChatListItemsCount = 0,
          searchContactListItemsCount = 0,
          chatListItem0 = document.querySelector('.chat-list-item-0'),
          contactListItem0 = document.querySelector('.contact-list-item-0'),
          searchChatListItems = [].slice.call(
            document.querySelectorAll('#chat-list li:not(.chat-contact-list-item-title)')
          ),
          searchContactListItems = [].slice.call(
            document.querySelectorAll('#contact-list li:not(.chat-contact-list-item-title)')
          );

        // Search in chats
        searchChatContacts(searchChatListItems, searchChatListItemsCount, searchValue, chatListItem0);
        // Search in contacts
        searchChatContacts(searchContactListItems, searchContactListItemsCount, searchValue, contactListItem0);
      });
    }

    // Search chat and contacts function
    function searchChatContacts(searchListItems, searchListItemsCount, searchValue, listItem0) {
      searchListItems.forEach(searchListItem => {
        let searchListItemText = searchListItem.textContent.toLowerCase();
        if (searchValue) {
          if (-1 < searchListItemText.indexOf(searchValue)) {
            searchListItem.classList.add('d-flex');
            searchListItem.classList.remove('d-none');
            searchListItemsCount++;
          } else {
            searchListItem.classList.add('d-none');
          }
        } else {
          searchListItem.classList.add('d-flex');
          searchListItem.classList.remove('d-none');
          searchListItemsCount++;
        }
      });
      // Display no search found if searchListItemsCount == 0
      if (searchListItemsCount == 0) {
        listItem0.classList.remove('d-none');
      } else {
        listItem0.classList.add('d-none');
      }
    }

    // Send Message
    if (formSendMessage) {
      formSendMessage.addEventListener('submit', e => {
        e.preventDefault();
        if (messageInput && messageInput.value) {
          let clienteIdElement = document.getElementById('cliente_id');
          const userIdElement = document.getElementById('user_id');

          // Se o elemento cliente_id não existir, crie-o dinamicamente
          if (!clienteIdElement && userIdElement) {
            clienteIdElement = document.createElement('input');
            clienteIdElement.type = 'hidden';
            clienteIdElement.id = 'cliente_id';
            clienteIdElement.value = userIdElement.value; // ou outro valor apropriado
            document.body.appendChild(clienteIdElement);
          }

          if (clienteIdElement || userIdElement) {
            // Enviar mensagem ao servidor
            fetch('/chat/send', {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
              },
              body: JSON.stringify({
                message: messageInput.value,
                cliente_id: clienteIdElement ? clienteIdElement.value : null,
                user_id: userIdElement ? userIdElement.value : null
              })
            })
            .then(response => response.json())
            .then(data => {
              // Verificar se a mensagem foi enviada com sucesso
              if (data) {
                // Criar um elemento div e adicionar uma classe
                let renderMsg = document.createElement('div');
                renderMsg.className = 'chat-message-text mt-2';
                renderMsg.innerHTML = '<p class="mb-0 text-break">' + data.message + '</p>';
                let lastChatMessageWrapper = document.querySelector('li:last-child .chat-message-wrapper');
                if (lastChatMessageWrapper) {
                  lastChatMessageWrapper.appendChild(renderMsg);
                } else {
                  // Criar um novo item de lista se não houver mensagens anteriores
                  let newChatMessage = document.createElement('li');
                  newChatMessage.className = 'chat-message chat-message-right';
                  const userAvatarImg = document.querySelector('.user-avatar img');
                  const userAvatarSrc = userAvatarImg ? userAvatarImg.src : '';

                  newChatMessage.innerHTML = `
                    <div class="d-flex overflow-hidden">
                      <div class="chat-message-wrapper flex-grow-1">
                        <div class="chat-message-text">
                          <p class="mb-0">${data.message}</p>
                        </div>
                        <div class="text-end text-muted mt-1">
                          <i class='ti ti-checks ti-xs me-1 text-success'></i>
                          <small>${new Date().toLocaleTimeString()}</small>
                        </div>
                      </div>
                      <div class="user-avatar flex-shrink-0 ms-3">
                        <div class="avatar avatar-sm">
                          <img src="${userAvatarSrc}" alt="Avatar" class="rounded-circle">
                        </div>
                      </div>
                    </div>
                  `;
                  document.querySelector('.chat-history').appendChild(newChatMessage);
                }
                messageInput.value = '';
                scrollToBottom();
              }
            })
            .catch(error => console.error('Erro ao enviar mensagem:', error));
          } else {
            console.error('Elemento cliente_id ou user_id não encontrado.');
          }
        }
      });
    }

    // on click of chatHistoryHeaderMenu, Remove data-overlay attribute from chatSidebarLeftClose to resolve overlay overlapping issue for two sidebar
    let chatHistoryHeaderMenu = document.querySelector(".chat-history-header [data-target='#app-chat-contacts']"),
      chatSidebarLeftClose = document.querySelector('.app-chat-sidebar-left .close-sidebar');
    chatHistoryHeaderMenu.addEventListener('click', e => {
      chatSidebarLeftClose.removeAttribute('data-overlay');
    });

    // Speech To Text
    if (speechToText.length) {
      var SpeechRecognition = SpeechRecognition || webkitSpeechRecognition;
      if (SpeechRecognition !== undefined && SpeechRecognition !== null) {
        var recognition = new SpeechRecognition(),
          listening = false;
        speechToText.on('click', function () {
          const $this = $(this);
          recognition.onspeechstart = function () {
            listening = true;
          };
          if (listening === false) {
            recognition.start();
          }
          recognition.onerror = function (event) {
            listening = false;
          };
          recognition.onresult = function (event) {
            $this.closest('.form-send-message').find('.message-input').val(event.results[0][0].transcript);
          };
          recognition.onspeechend = function (event) {
            listening = false;
            recognition.stop();
          };
        });
      }
    }

    // Select contatos
    const contactListItems = document.querySelectorAll('#contact-list .chat-contact-list-item');
    contactListItems.forEach(contactListItem => {
      contactListItem.addEventListener('click', e => {
        // Remove active class from all contact list items
        contactListItems.forEach(item => item.classList.remove('active'));
        // Add active class to the clicked contact list item
        e.currentTarget.classList.add('active');
        
        // Obter o id do cliente
        const clienteId = e.currentTarget.dataset.clienteId;
        
        // Abrir o histórico de chat
        const chatHistoryBody = document.querySelector('.chat-history-body');
        chatHistoryBody.classList.remove('d-none'); // Certifique-se de que o chat está visível
        console.log('Cliente ID:', clienteId);
        // Fetch messages for the selected contact
        fetchMessages(clienteId);
      });
    });

    // Função para buscar mensagens
    function fetchMessages(clienteId) {
      fetch(`/chat/messages/${clienteId}`)
        .then(response => {
          if (!response.ok) {
            throw new Error('Erro na rede: ' + response.statusText);
          }
          return response.json();
        })
        .then(messages => {
          const chatHistory = document.querySelector('.chat-history');
          chatHistory.innerHTML = ''; // Limpa o histórico atual de chat
          messages.forEach(message => {
            const messageElement = document.createElement('div');
            messageElement.className = 'chat-message';
            messageElement.innerHTML = `
              <div class="d-flex overflow-hidden">
                <div class="chat-message-wrapper flex-grow-1">
                  <div class="chat-message-text">
                    <p class="mb-0">${message.message}</p>
                  </div>
                  <div class="text-end text-muted mt-1">
                    <small>${new Date(message.created_at).toLocaleTimeString()}</small>
                  </div>
                </div>
              </div>
            `;
            chatHistory.appendChild(messageElement);
          });
          scrollToBottom();
        })
        .catch(error => console.error('Erro ao buscar mensagens:', error));
    }

  })();
});